package de.brightbyte.db;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Map;

public class CursorBasedInserter extends AbstractInserter {

	public static final InserterFactory factory = new InserterFactory() {
		public Inserter newInserter(DatabaseTable table, String automaticField, int bufferLength, int bufferWidth) throws SQLException {
			return new CursorBasedInserter(table, automaticField);
		}
	};

	protected Statement statement;
	protected ResultSet cursor;
	protected String select;
	
	public CursorBasedInserter(DatabaseTable table, String automaticField) {
		super(table, automaticField);
		
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT ");
		sql.append(" * ");
		
		/*
		boolean first = true;
		for (DatabaseField field : fields.values()) {
			if (first) first= false;
			else sql.append(", ");
			
			sql.append(field.getName());
		}
		*/
		sql.append(" FROM ");
		sql.append(table.getSQLName());
		sql.append(" LIMIT 0");
		
		init(sql.toString());
	}

	public CursorBasedInserter(DatabaseAccess access, Map<String, DatabaseField> fields, String automaticField, String select) {
		super(access, fields, automaticField);
		init(select);
	}
	
	private void init(String select) {
		//if (sensitive) cursorType = ResultSet.TYPE_FORWARD_ONLY | ResultSet.TYPE_SCROLL_SENSITIVE;
		//else cursorType = ResultSet.TYPE_FORWARD_ONLY | ResultSet.TYPE_SCROLL_SENSITIVE;
			
		this.select = select;
	}
	
	public void connect() throws SQLException {
		if (statement!=null) return;		
		Connection conn = table.getAccess().getConnection();
		
		statement = conn.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE);
		statement.setEscapeProcessing(false);

		access.trace("SQL: "+select);
		cursor = statement.executeQuery(select);
		cursor.moveToInsertRow();
	}

	@Override
	public void close() {
		try {
			if (cursor!=null) cursor.close();
		} catch (SQLException e) { /* ignore */ }

		try {
			if (statement!=null) statement.close();
			statement = null;
		} catch (SQLException e) { /* ignore */ }
	}

	@Override
	public void flush() throws SQLException {
		//noop
	}

	@Override
	public void insert(Map<String, Object> data) throws SQLException {
		for (DatabaseField field : fields.values()) {
			String k = field.getName();
			Object v = data.get(k);
			
			if (v!=null) updateObject(k, v);
		}
		
		updateRow();
	}

	@Override
	public void updateRow() throws SQLException {
		if (cursor==null) connect();
		
		int autoId = -1;
		
		if (automaticField!=null) {
			autoId = getNextId(automaticField);
			if (autoId<=0) 
				throw new IllegalStateException("failed to fetch next id!");
			
			updateInt(automaticField, autoId);
			//System.out.println(table.getName()+": automatic id "+autoId);
		}
		
		/*
		for (DatabaseField field : fields.values()) {
			String k = field.getName();
			
			try {
				Object v = cursor.getObject(k); //FIXME: returns dummy values!
				
				if (v==null && field.isRequired()) {
					if (field.getKeyType() == KeyType.PRIMARY && getNextId()>0) v = getNextId();
					else throw new SQLException("missing value for required field "+k);
				}
				
				if (field.getKeyType() == KeyType.PRIMARY && v instanceof Number) {
					lastId = Math.max(lastId, ((Number)v).intValue());
				}
				
				//updateObject(k, v);
			}
			catch (SQLException ex) {
				throw ex;
			}
		}
		*/
		
		cursor.insertRow();
		cursor.moveToInsertRow();

		if (autoId>0) {
			incNextId(autoId);
		}
	}
	
	@Override
	public void updateBigDecimal(String columnName, BigDecimal x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateBigDecimal(columnName, x);
	}

	@Override
	public void updateBlob(String columnName, Blob x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateBlob(columnName, x);
	}

	@Override
	public void updateBoolean(String columnName, boolean x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateBoolean(columnName, x);
	}

	@Override
	public void updateByte(String columnName, byte x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateByte(columnName, x);
	}

	@Override
	public void updateBytes(String columnName, byte[] x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateBytes(columnName, x);
	}

	@Override
	public void updateClob(String columnName, Clob x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateClob(columnName, x);
	}

	@Override
	public void updateDate(String columnName, Date x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateDate(columnName, x);
	}

	@Override
	public void updateDouble(String columnName, double x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateDouble(columnName, x);
	}

	@Override
	public void updateFloat(String columnName, float x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateFloat(columnName, x);
	}

	@Override
	public void updateInt(String columnName, int x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateInt(columnName, x);
	}

	@Override
	public void updateLong(String columnName, long x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateLong(columnName, x);
	}

	@Override
	public void updateNull(String columnName) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateNull(columnName);
	}

	@Override
	public void updateObject(String columnName, Object x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateObject(columnName, x);
	}

	@Override
	public void updateShort(String columnName, short x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateShort(columnName, x);
	}

	@Override
	public void updateString(String columnName, String x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateString(columnName, x);
	}

	@Override
	public void updateTime(String columnName, Time x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateTime(columnName, x);
	}

	@Override
	public void updateTimestamp(String columnName, Timestamp x) throws SQLException {
		if (cursor==null) connect();
		
		cursor.updateTimestamp(columnName, x);
	}

	@Override
	public void setLenient(boolean lenient) {
		if (lenient==true) throw new IllegalArgumentException("lenient operation not supported");
	}

	
}
