package de.brightbyte.db;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;

public abstract class BufferedInserterBase extends AbstractInserter {
	protected Map<String, Object> insertRow = new HashMap<String, Object>();
	protected boolean failed = false;
	
	public BufferedInserterBase(DatabaseTable table, String automaticField) {
		super(table, automaticField);
	}

	@Override
	public void insert(Map<String, Object> data) throws SQLException {
		insertRow.clear();
		insertRow.putAll(data);
		updateRow();
	}

	@Override
	public void updateRow() throws SQLException {
		if (failed) throw new SQLException("inserter invalidated by previous error in background task");
		
		int autoId = -1;
		
		if (automaticField!=null) {
			Number v = (Number)currentValue(automaticField);
			if (v!=null) {
				autoId = v.intValue();
			}
			else {
				autoId = getNextId(automaticField);
				if (autoId<=0) 
					throw new IllegalStateException("failed to fetch next id!");
				
				updateInt(automaticField, autoId);
				//System.out.println(table.getName()+": automatic id "+autoId);
			}
		}
				
		pushRow();
		insertRow.clear();

		if (autoId>0) {
			incNextId(autoId);
		}
	}
	
	protected abstract void pushRow() throws SQLException;
	
	protected Object currentValue(String columnName) {
		return insertRow.get(columnName);
	}
	
	@Override
	public void updateObject(String columnName, Object x) throws SQLException {
		insertRow.put(columnName, x);
	}
	
	@Override
	public void updateBigDecimal(String columnName, BigDecimal x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateBlob(String columnName, Blob x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateBoolean(String columnName, boolean x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateByte(String columnName, byte x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateBytes(String columnName, byte[] x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateClob(String columnName, Clob x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateDate(String columnName, Date x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateDouble(String columnName, double x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateFloat(String columnName, float x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateInt(String columnName, int x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateLong(String columnName, long x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateNull(String columnName) throws SQLException {
		updateObject(columnName, null);
	}

	@Override
	public void updateShort(String columnName, short x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateString(String columnName, String x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateTime(String columnName, Time x) throws SQLException {
		updateObject(columnName, x);
	}

	@Override
	public void updateTimestamp(String columnName, Timestamp x) throws SQLException {
		updateObject(columnName, x);
	}
	
}
