package de.brightbyte.db;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

public class BufferBasedInserter extends BufferedInserterBase {

	public static final InserterFactory factory = new InserterFactory() {
		public Inserter newInserter(DatabaseTable table, String automaticField, int bufferLength, int bufferWidth) throws SQLException {
			return new BufferBasedInserter(table, automaticField, bufferLength * bufferWidth);
		}
	};

	private Statement statement;
	private PreparedStatement auxStatement;
	private StringBuilder buffer = new StringBuilder();
	private String insertPrefix; 
	private String[] params;
	
	private int maxBufferSize;

	public BufferBasedInserter(DatabaseTable table, String automaticField, int maxBufferSize) throws SQLException {
		super(table, automaticField);
		this.maxBufferSize = Math.min(maxBufferSize, table.getAccess().getMaxStatementSize() / 2); //XXX: hack! dividing by 2 is DIRTY!
	}
	
	private StringBuilder lineBuffer = new StringBuilder();

	@Override
	protected void pushRow() throws SQLException {
		if (auxStatement==null) connect();
		
		lineBuffer.setLength(0);		
		lineBuffer.append('(');
		
		for (int i = 0; i < params.length; i++) {
			if (i>0) lineBuffer.append(',');
			
			Object v = insertRow.get(params[i]);
			lineBuffer.append(access.encodeValue(v));
		}
		lineBuffer.append(')');
		
		if (insertPrefix.length()+buffer.length()+lineBuffer.length()+1 > maxBufferSize) {
			flushInBackground();
		}
		
		if (insertPrefix.length()+buffer.length()+lineBuffer.length()+1 > maxBufferSize) {
			access.warn("data-chunk too large ("+lineBuffer.length()+" characters). bypassing buffer."); 
			
			//copy insert row
			final Object[] data = new Object[params.length];
			for (int i = 0; i < params.length; i++) {
				data[i] = insertRow.get(params[i]);
			}
			
			//NOTE: needs top be synced with background query queue
			DatabaseTask task = new DatabaseTask() {
				public Object call() throws SQLException {
					access.trace("executing single-row insert"); 
					
					for (int i = 0; i < data.length; i++) {
						Object v = data[i];
						auxStatement.setObject(i+1, v);
					}

					return auxStatement.execute();
				}
			
			};
			
			access.executeSynced(task);
		}
		else {
			if (buffer.length()>0) buffer.append(',');
			buffer.append(lineBuffer);
		}
	}

	protected void connect() throws SQLException {
		if (statement!=null) return;		
		Connection conn = table.getAccess().getConnection();
		
		params = new String[fields.size()];
		int i = 0;
		for (DatabaseField field : fields.values()) {
			params[i++] = field.getName();
		}
		
		StringBuilder sql = new StringBuilder();
		sql.append("INSERT ");
		if (lenient) sql.append("IGNORE "); 
		sql.append("INTO ");
		sql.append(access.quoteName(table.getSQLName()));
		sql.append(" ( ");
		for (int j = 0; j < params.length; j++) {
			if (j>0) sql.append(", ");
			sql.append(access.quoteName(params[j]));
		}
		sql.append(" ) ");
		sql.append(" VALUES ");
		
		insertPrefix = sql.toString();
		statement = conn.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		statement.setEscapeProcessing(false);
		
		sql.append(" ( ");
		for (int j = 0; j < params.length; j++) {
			if (j>0) sql.append(", ");
			sql.append("? ");
		}
		sql.append(" ) ");
		auxStatement = conn.prepareStatement(sql.toString(), ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
	}

	@Override
	public void close() {
		if (buffer.length()>0) {
			access.warn(buffer.length()+" characters for table "+table.getName()+" discard on close! call flush() first if possible."); //FIXME: logger!
		}
			
		try {
			if (statement!=null) statement.close();
			statement = null;
		} catch (SQLException e) { /* ignore */ }

		try {
			if (auxStatement!=null) auxStatement.close();
			auxStatement = null;
		} catch (SQLException e) { /* ignore */ }
	}

	protected void flushInBackground() throws SQLException {
		DatabaseTask flushTask = getFlushTask();
		
		if (flushTask!=null) {
			access.trace("scheduling flush");
			access.executeLater(flushTask);
		}
	}
	
	@Override
	public void flush() throws SQLException {
		DatabaseTask flushTask = getFlushTask();
		if (flushTask!=null) {
			access.trace("schduling flush & wait"); 
			access.executeSynced(flushTask);
		}
	}

	protected DatabaseTask getFlushTask() {
		if (buffer.length()>0) {
			final String sql = insertPrefix+buffer; 
			buffer.setLength(0);
			
			return new DatabaseTask() {
				public Object call() throws SQLException {
					try {
						access.info("flushing "+sql.length()+" characters into table "+table.getName()); 
						if (statement==null) connect();		
						return statement.executeUpdate(sql);
					}
					catch (SQLException ex) {
						/*
						try {
							IOUtil.spit(new java.io.File("/tmp/bad.sql"), sql, "UTF-8", false);
						} catch (IOException e) {
							e.printStackTrace();
						}
						*/
						throw ex; //merely a debug hook
					}
				}
			
			};
		}
		else return null;
	}

}
