package de.brightbyte.db;

import java.sql.SQLException;

public class BatchBasedInserter extends StatementBasedInserter {

	public static final InserterFactory factory = new InserterFactory() {
		public Inserter newInserter(DatabaseTable table, String automaticField, int bufferLength, int bufferWidth) throws SQLException {
			return new BatchBasedInserter(table, automaticField, bufferLength);
		}
	};
	
	protected int batchSize = 0;
	protected int maxBatchSize;

	public BatchBasedInserter(DatabaseTable table, String automaticField, int maxBatchSize) {
		super(table, automaticField);
		this.maxBatchSize = maxBatchSize;
	}

	@Override
	protected void pushRow() throws SQLException {
		if (statement==null) connect();		
		
		for (int i = 0; i < params.length; i++) {
			Object v = insertRow.get(params[i]);
			statement.setObject(i+1, v);
		}
		
		statement.addBatch();
		batchSize++;
		
		if (batchSize>=maxBatchSize) {
			flush();
		} 
	}

	@Override
	public void close() {
		if (batchSize>0) {
			access.warn(batchSize+" entries for table "+table.getName()+" discard on close! call flush() first if possible."); 
		}

		super.close();
	}

	@Override
	public void flush() throws SQLException {
		if (batchSize>0) {
			access.info("flushing "+batchSize+" rows into table "+table.getName()); 

			if (statement==null) connect();		
			statement.executeBatch();
			batchSize = 0;
		}
	}

}
