package de.brightbyte.db;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Map;

public abstract class AbstractInserter implements Inserter {
	protected Map<String, DatabaseField> fields;
	protected DatabaseTable table;
	protected DatabaseAccess access;
	protected boolean lenient;
	
	protected String automaticField;
	protected int nextId = -1;
	protected int lastId = -1;
	
	protected AbstractInserter(DatabaseAccess access, Map<String, DatabaseField> fields, String automaticField) {
		this.access = access;
		this.fields = fields;
		this.automaticField = automaticField;
	}
	
	public void disableAutomaticField() {
		automaticField = null;
	}
	
	protected AbstractInserter(DatabaseTable table, String automaticField) {
		this(table.getAccess(), table.getFields(), automaticField);
		
		this.table = table;
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#getLastId()
	 */
	public int getLastId() throws SQLException {
		if (lastId<0 && table!=null && automaticField!=null) {
			initLastId(automaticField);
		}
		
		return lastId;
	}
	
	protected void initLastId(String idField) throws SQLException {
		if (lastId<0 && table!=null) {
			DatabaseField f = table.getField(idField);
			if (f!=null) {
				String sql = "SELECT MAX("+f.getName()+") FROM "+table.getSQLName();
				Number n = (Number)access.executeSingleValueQuery("max id", sql);
				
				lastId = n==null ? 0 : n.intValue();
			}
		}
	}
	
	protected int getNextId(String idField) throws SQLException {
		if (nextId<0 && table!=null) {
			initLastId(idField);
		
			if (lastId>=0) incNextId(lastId);
		}
		
		return nextId;
	}

	protected void incNextId(int last) {
		lastId = Math.max(last, lastId);
		nextId = lastId + 1;
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#flush()
	 */
	public abstract void flush() throws SQLException;
	
	/* (non-Javadoc)
	 * 
	 * @see de.brightbyte.db.Inserter#close()
	 */
	public abstract void close();
	
	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#insert(java.util.Map)
	 */
	public abstract void insert(Map<String, Object> data) throws SQLException;
	
	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateRow()
	 */
	public abstract void updateRow() throws SQLException;
	
	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateBigDecimal(java.lang.String, java.math.BigDecimal)
	 */
	public abstract void updateBigDecimal(String columnName, BigDecimal x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateBlob(java.lang.String, java.sql.Blob)
	 */
	public abstract void updateBlob(String columnName, Blob x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateBoolean(java.lang.String, boolean)
	 */
	public abstract void updateBoolean(String columnName, boolean x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateByte(java.lang.String, byte)
	 */
	public abstract void updateByte(String columnName, byte x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateBytes(java.lang.String, byte[])
	 */
	public abstract void updateBytes(String columnName, byte[] x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateClob(java.lang.String, java.sql.Clob)
	 */
	public abstract void updateClob(String columnName, Clob x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateDate(java.lang.String, java.sql.Date)
	 */
	public abstract void updateDate(String columnName, Date x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateDouble(java.lang.String, double)
	 */
	public abstract void updateDouble(String columnName, double x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateFloat(java.lang.String, float)
	 */
	public abstract void updateFloat(String columnName, float x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateInt(java.lang.String, int)
	 */
	public abstract void updateInt(String columnName, int x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateLong(java.lang.String, long)
	 */
	public abstract void updateLong(String columnName, long x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateNull(java.lang.String)
	 */
	public abstract void updateNull(String columnName) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateObject(java.lang.String, java.lang.Object)
	 */
	public abstract void updateObject(String columnName, Object x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateShort(java.lang.String, short)
	 */
	public abstract void updateShort(String columnName, short x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateString(java.lang.String, java.lang.String)
	 */
	public abstract void updateString(String columnName, String x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateTime(java.lang.String, java.sql.Time)
	 */
	public abstract void updateTime(String columnName, Time x) throws SQLException;

	/* (non-Javadoc)
	 * @see de.brightbyte.db.Inserter#updateTimestamp(java.lang.String, java.sql.Timestamp)
	 */
	public abstract void updateTimestamp(String columnName, Timestamp x) throws SQLException;

	public DatabaseTable getTable() {
		return table;
	}

	public boolean isLenient() {
		return lenient;
	}

	public void setLenient(boolean lenient) {
		this.lenient = lenient;
	}
	
}
